{ ARS Traffic & Transport Technology }
{ Project : uGoogleMaps.pas                                               }
{ Comment : Google Maps API wrapper                                       }
{                                                                         }
{    Date : 14 mrt 2008                                                   }
{  Author : Wouter van Nifterick                                          }
{}
{ The aim of this unit is to wrap the Google Maps API in such a way that  }
{ users don't need to know that map operations end up being rendered by   }
{ a browser component and JavaScript.                                     }
{                                                                         }
{ JavaScript classes have a Delphi counterpart, and each Delphi class     }
{ takes care of proper JavaScript rendering and execution.                }
{                                                                         }
{ For many things, like constructing detailed polygons, this provides a   }
{ major performance boost and adds compile-time type checking.            }
{                                                                         }
{ A big limitation so far is that I didn't find a way to directly pass    }
{ complex types from and to the JavaScript engine in IE via COM, so for   }
{ now, everything needs to be (de)serialized to and from strings. :(      }
{}
{ Last modified                                                           }
{    Date :                                                               }
{  Author :                                                               }
{}

{$M+}

unit uGoogleMaps;

interface

uses
  Controls,
  Dialogs,
  ActiveX,
  StdCtrls,
  ExtCtrls,
  SysUtils,
  Classes,
  Contnrs,
  Forms,
  SHDocVw,
  MSHTML,
  StrUtils,
  DouglasPeuckers
//  ,  uGoogleEarth_intf
  ;

const
  GoogleMapsFileName = 'GoogleMaps.html';
  WGS84_MULT_FACT    = 100000; // multiply lat/lon values by this value in order to fit them into integers
  DEFAULT_SIMPLIFY_TOLERANCE = 0.5;

{$R GoogleMaps_html.res}


type


  TGoogleMapControl = (MC_NONE=1,MC_SMALL,MC_LARGE);
  TGoogleMapType    = (MT_NORMAL=1,MT_SATELLITE,MT_HYBRID,MT_PHYSICAL,MT_SATELLITE_3D);

  TGoogleMaps       = class; // forward declaration

  GIcon             = class end; // to be implemented

  IJsClassWrapper=interface(IInterface)
    function JsClassName:String;

    function GetJsVarName:String;
    procedure SetJsVarName(const aVarName:String);
    property JsVarName:String read GetJsVarName write SetJsVarName;

    function ToJavaScript:String;
  end;

  IHidable=interface(IInterface)
    procedure hide;                         // Hides the object if the overlay is both currently visible and the overlay's supportsHide() method returns true. Note that this method will trigger the respective visibilitychanged event for each child overlay that fires that event (e.g. GMarker.visibilitychanged, GGroundOverlay.visibilitychanged, etc.). If no overlays are currently visible that return supportsHide() as true, this method has no effect. (Since 2.87)
    function  isHidden           : Boolean; // Returns true if the GGeoXml object is currently hidden, as changed by the GGeoXml.hide() method. Otherwise returns false. (Since 2.87)
    procedure show;                         // Shows the child overlays created by the GGeoXml object, if they are currently hidden. Note that this method will trigger the respective visibilitychanged event for each child overlay that fires that event (e.g. GMarker.visibilitychanged, GGroundOverlay.visibilitychanged). (Since 2.87)
    function  supportsHide       : Boolean; // 
  end;


  // marker class
  GMarkerOptions=record
    icon          : GIcon;    // Chooses the Icon for this class. If not specified, G_DEFAULT_ICON is used. (Since 2.50)
    dragCrossMove : Boolean;  // When dragging markers normally, the marker floats up and away from the cursor. Setting this value to true keeps the marker underneath the cursor, and moves the cross downwards instead. The default value for this option is false. (Since 2.63)
    title         : String;   // This string will appear as tooltip on the marker, i.e. it will work just as the title attribute on HTML elements. (Since 2.50)
    clickable     : Boolean;  // Toggles whether or not the marker is clickable. Markers that are not clickable or draggable are inert, consume less resources and do not respond to any events. The default value for this option is true, i.e. if the option is not specified, the marker will be clickable. (Since 2.50)
    draggable     : Boolean;  // Toggles whether or not the marker will be draggable by users. Markers set up to be dragged require more resources to set up than markers that are clickable. Any marker that is draggable is also clickable, bouncy and auto-pan enabled by default. The default value for this option is false. (Since 2.61)
    bouncy        : Boolean;  // Toggles whether or not the marker should bounce up and down after it finishes dragging. The default value for this option is false. (Since 2.61)
    bounceGravity : Integer;   // When finishing dragging, this number is used to define the acceleration rate of the marker during the bounce down to earth. The default value for this option is 1. (Since 2.61)
    autoPan       : Boolean;  // Auto-pan the map as you drag the marker near the edge. If the marker is draggable the default value for this option is true. (Since 2.87)

    // to implement:
    //    zIndexProcess : Function; // This function is used for changing the z-Index order of the markers when they are overlaid on the map and is also called when their infowindow is opened. The default order is that the more southerly markers are placed higher than more northerly markers. This function is passed in the GMarker object and returns a number indicating the new z-index. (Since 2.98)
  end;

  TGPoint=class

  end;

  TGLatLng=class(TInterfacedObject,IJsClassWrapper)
  private
    FLat,
    FLng:Double;
    FJsVarName: String;
    function GetJsVarName: String;
    procedure SetJsVarName(const Value: String);
  published
    constructor Create(aLat,aLng:Double);
    property Lat:Double read FLat write FLat;
    property Lng:Double read FLng write FLng;
    function ToJavaScript:String;
    function Equals(const AGLatLng:TGLatLng):Boolean;
    function ToString:String;
    function JsClassName:String;virtual;
    property JsVarName:String read GetJsVarName write SetJsVarName;
  end;


  TGBounds=class(TInterfacedObject,IJsClassWrapper)
  private
    FJsVarName: String;
    FMinX, FMinY, FMaxX, FMaxY:Double;
    FMin,FMax,FMid:TGLatLng;
    function GetMax: TGLatLng;
    function GetMid: TGLatLng;
    function GetMin: TGLatLng;
    procedure SetJsVarName(const Value: String);
    function GetJsVarName: String;
  published
    destructor Destroy;override;
    property minX : Double read FMinX write FMinX;
    property minY : Double read FMinY write FMinY;
    property maxX : Double read FMaxX write FMaxX;
    property maxY : Double read FMaxY write FMaxY;
    function ToString:String;
    function Equals(aGBounds:TGBounds):Boolean;
    property Min:TGLatLng read GetMin;
    property Mid:TGLatLng read GetMid;
    property Max:TGLatLng read GetMax;
    function JsClassName:String;virtual;
    property JsVarName:String read GetJsVarName write SetJsVarName;
    function ToJavaScript:String;
  end;

  TGLatLngBounds=class
  private
    procedure setNorthEast(const Value: TGLatLng);
    procedure setSouthWest(const Value: TGLatLng);
  published
    constructor Create(sw,ne:TGLatLng);
    destructor Destroy;override;
    function contains(aLatLng:TGLatLng):Boolean; deprecated; // Returns true iff the geographical coordinates of the point lie within this rectangle. (Deprecated since 2.88)
    function containsLatLng(aLatLng:TGLatLng):Boolean; // Returns true iff the geographical coordinates of the point lie within this rectangle. (Since 2.88)
    function intersects(aGLatLngBounds:TGLatLngBounds):Boolean;
    function containsBounds(aGLatLngBounds:TGLatLngBounds):Boolean;
    procedure extend(aLatLng:TGLatLng); // Enlarges this rectangle such that it contains the given point. In longitude direction, it is enlarged in the smaller of the two possible ways. If both are equal, it is enlarged at the eastern boundary.

    function toSpan()       :	TGLatLng; //	Returns a GLatLng whose coordinates represent the size of this rectangle.
    function isFullLat()    :	Boolean ; //	Returns true if this rectangle extends from the south pole to the north pole.
    function isFullLng()    :	Boolean ; //	Returns true if this rectangle extends fully around the earth in the longitude direction.
    function isEmpty()      :	Boolean ; //	Returns true if this rectangle is empty.
    function getCenter()    :	TGLatLng; //	Returns the point at the center of the rectangle. (Since 2.52)

    function getSouthWest() :	TGLatLng; //	Returns the point at the south-west corner of the rectangle.
    function getNorthEast() :	TGLatLng; //	Returns the point at the north-east corner of the rectangle.
    property SouthWest : TGLatLng read getSouthWest write setSouthWest;
    property NorthEast : TGLatLng read getNorthEast write setNorthEast;

    function ToString:String;
    function Equals(aGLatLngBounds:TGLatLngBounds):Boolean;
  end;

  TColor = integer;

  // abstract class.. subclassed by TGMarker and TGPolygon and TGPolyLine..

  TGOverlay=class(TInterfacedObject,IJsClassWrapper,IHidable)
  private
    FID: Integer;
    FGoogleMaps: TGoogleMaps;
    FName: String;
    FJsVarName:String;
    procedure SetID(const Value: Integer);
    procedure SetGoogleMaps(const Value: TGoogleMaps);
    procedure SetName(const Value: String);
    function GetJsVarName: String;
    procedure SetJsVarName(const Value: String);
  public
    procedure hide;virtual;
    function isHidden: Boolean;virtual;
    procedure show;virtual;
    function supportsHide: Boolean;virtual;
  published
    property ID:Integer read FID write SetID;
    function ToJavaScript:String;virtual;abstract;
    property JsVarName:String read GetJsVarName write SetJsVarName;
    property GoogleMaps:TGoogleMaps read FGoogleMaps write SetGoogleMaps;
    property Name:String read FName write SetName;
    function JsClassName:string;virtual;
  end;

  TOverlayList=class(TObjectList)
  private
    AutoIncrementID:Integer;
    function GetItems(Index: Integer): TGOverlay;
    procedure SetItems(Index: Integer; const Value: TGOverlay);
  public
    property Items[Index:Integer]:TGOverlay read GetItems write SetItems; default;
  published
    constructor Create;
    destructor Destroy;override;
    function Add(aGOverlay:TGOverlay):Integer;
    procedure Clear;override;
    function ToString:String;
  end;



  TGInfoWindow=class(TGOverlay,IJsClassWrapper,IHidable)
    procedure Maximize;
    procedure Restore;
  private
    FHTML: String;
    procedure SetHTML(const Value: String);
  public
    property HTML:String read FHTML write SetHTML;
    function JsClassName:String;override;
    constructor Create(const aCenter:TGLatLng);
    destructor Destroy;override;
    function ToJavaScript:String;override;
    function supportsHide: Boolean;override;
  end;


  // used to show a location on a map
  // can be dragged, can show a popup, can have custom colors and icon
  TGMarker=class(TGOverlay,IJsClassWrapper,IHidable)
  private
    FCenter: TGLatLng;
    FDraggingEnabled: Boolean;
    procedure setLatLng(const Value: TGLatLng);
    procedure SetDraggingEnabled(const Value: Boolean);
  public
    function supportsHide: Boolean;override;
  published
    function JsClassName:String;override;
    constructor Create(const aCenter:TGLatLng);
    destructor Destroy;override;
    property Center:TGLatLng read FCenter write setLatLng;
    property DraggingEnabled:Boolean read FDraggingEnabled write SetDraggingEnabled;
    function ToJavaScript:String;override;
   { TODO 3 -oWouter : implement all marker methods and events }

    procedure openInfoWindow(aContent:String); // Opens the map info window over the icon of the marker. The content of the info window is given as a DOM node. Only option GInfoWindowOptions.maxWidth is applicable.
    procedure openInfoWindowHtml(aContent:String); // Opens the map info window over the icon of the marker. The content of the info window is given as a string that contains HTML text. Only option GInfoWindowOptions.maxWidth is applicable.
{    procedure openInfoWindowTabs(tabs, opts?) : none; // Opens the tabbed map info window over the icon of the marker. The content of the info window is given as an array of tabs that contain the tab content as DOM nodes. Only options GInfoWindowOptions.maxWidth and InfoWindowOptions.selectedTab are applicable.
    procedure openInfoWindowTabsHtml(tabs, opts?) : none; // Opens the tabbed map info window over the icon of the marker. The content of the info window is given as an array of tabs that contain the tab content as Strings that contain HTML text. Only options InfoWindowOptions.maxWidth and InfoWindowOptions.selectedTab are applicable.
    procedure bindInfoWindow(content, opts?) : none; // Binds the given DOM node to this marker. The content within this node will be automatically displayed in the info window when the marker is clicked. Pass content as null to unbind. (Since 2.85)
    procedure bindInfoWindowHtml(content, opts?) : none; // Binds the given HTML to this marker. The HTML content will be automatically displayed in the info window when the marker is clicked. Pass content as null to unbind. (Since 2.85)
    procedure bindInfoWindowTabs(tabs, opts?) : none; // Binds the given GInfoWindowTabs (provided as DOM nodes) to this marker. The content within these tabs' nodes will be automatically displayed in the info window when the marker is clicked. Pass tabs as null to unbind. (Since 2.85)
    procedure bindInfoWindowTabsHtml(tabs, opts?) : none; // Binds the given GInfoWindowTabs (provided as strings of HTML) to this marker. The HTML content within these tabs will be automatically displayed in the info window when the marker is clicked. Pass tabs as null to unbind. (Since 2.85)
    procedure closeInfoWindow() : none; // Closes the info window only if it belongs to this marker. (Since 2.85)
    procedure showMapBlowup(opts?) : none; // Opens the map info window over the icon of the marker. The content of the info window is a closeup map around the marker position. Only options InfoWindowOptions.zoomLevel and InfoWindowOptions.mapType are applicable.
    procedure getIcon() : GIcon; // Returns the icon of this marker, as set by the constructor.
    procedure getTitle() : String; // Returns the title of this marker, as set by the constructor via the GMarkerOptions.title property. Returns undefined if no title is passed in. (Since 2.85)
    procedure getPoint() : GLatLng; // Returns the geographical coordinates at which this marker is anchored, as set by the constructor or by setPoint(). (Deprecated since 2.88)
    procedure getLatLng() : GLatLng; // Returns the geographical coordinates at which this marker is anchored, as set by the constructor or by setLatLng(). (Since 2.88)
    procedure setPoint(latlng) : none; // Sets the geographical coordinates of the point at which this marker is anchored. (Deprecated since 2.88)
    procedure setLatLng(latlng) : none; // Sets the geographical coordinates of the point at which this marker is anchored. (Since 2.88)
    procedure enableDragging() : none; // Enables the marker to be dragged and dropped around the map. To function, the marker must have been initialized with GMarkerOptions.draggable = true.
    procedure disableDragging() : none; // Disables the marker from being dragged and dropped around the map.
    procedure draggable() : Boolean; // Returns true if the marker has been initialized via the constructor using GMarkerOptions.draggable = true. Otherwise, returns false.
    procedure draggingEnabled() : Boolean; // Returns true if the marker is currently enabled for the user to drag on the map.
    procedure setImage(url) : none; // Requests the image specified by the url to be set as the foreground image for this marker. Note that neither the print image nor the shadow image are adjusted. Therefore this method is primarily intended to implement highlighting or dimming effects, rather than drastic changes in marker's appearances. (Since 2.75)
}

  end;


  TGGeoXml=class(TGOverlay,IJsClassWrapper,IHidable)
  private
    FUrlOfXml: String;
    procedure SetUrlOfXml(const Value: String);
  published

//    function  getTileLayerOverlay: GTileLayerOverlay; // GGeoXml objects may create a tile overlay for optimization purposes in certain cases. This method returns this tile layer overlay (if available). Note that the tile overlay may be null if not needed, or if the GGeoXml file has not yet finished loading. (Since 2.84)
//    function  getDefaultCenter   : GLatLng;           // Returns the center of the default viewport as a lat/lng. This function should only be called after the file has been loaded. (Since 2.84)
//    function  getDefaultSpan     : GLatLng;           // Returns the span of the default viewport as a lat/lng. This function should only be called after the file has been loaded. (Since 2.84)
//    function  getDefaultBounds   : GLatLngBounds;     // Returns the bounding box of the default viewport. This function should only be called after the file has been loaded. (Since 2.84)
    procedure gotoDefaultViewport(Map:TGoogleMaps);     // Sets the map's viewport to the default viewport of the XML file. (Since 2.84)
//    function  hasLoaded          : Boolean; // Checks to see if the XML file has finished loading, in which case it returns true. If the XML file has not finished loading, this method returns false. (Since 2.84)
//    function  loadedCorrectly    : Boolean; // Checks to see if the XML file has loaded correctly, in which case it returns true. If the XML file has not loaded correctly, this method returns false. If the XML file has not finished loading, this method's return value is undefined. (Since 2.84)
    function  supportsHide       : Boolean; override; // Always returns true. (Since 2.87)

    function JsClassName:String;override;
    constructor Create(const aUrlOfXml:String);
    destructor Destroy;override;
    property UrlOfXml:String read FUrlOfXml write SetUrlOfXml;
    function ToJavaScript:String;override;
  end;


  // polygon class
  TGPolygon=class(TGOverlay,IJsClassWrapper,IHidable)
  private
    FCoordinates:Array of TGLatLng;
    FOpacity: double;
    FWeightPx: integer;
    FColor: TColor;
    FSimplified: TGPolygon;
    FIsDirty: Boolean;
    procedure SetColor(const Value: TColor);
    procedure SetOpacity(const Value: double);
    procedure SetWeightPx(const Value: integer);
    function GetCount: Integer;
    procedure SetSimplified(const Value: TGPolygon);
    function GetSimplified: TGPolygon;
  public
    constructor Create(const aCoordinates: array of TGLatLng);overload;
    constructor Create(const aPoints:Array of TPointFloat2D);overload;
    function supportsHide: Boolean;override;
  published
    function JsClassName:String;override;
    procedure Clear;
    function ToJavaScript:String;override;
    function AddPoint(const aGLatLng:TGLatLng):integer;
    property Color:TColor read FColor write SetColor;
    property WeightPx:integer read FWeightPx write SetWeightPx;
    property Opacity:double read FOpacity write SetOpacity;// number between 0 and 1
    property Count:Integer read GetCount;
    destructor Destroy;override;
    property IsDirty:Boolean read FIsDirty write FIsDirty;
    property Simplified:TGPolygon read GetSimplified write SetSimplified;
    function getSimplifiedVersion(Tolerance:Double=DEFAULT_SIMPLIFY_TOLERANCE):TGPolygon;
    class function PolyTypeStr: String;virtual;
  end;

  TGPolyLine=class(TGPolygon,IJsClassWrapper,IHidable)
  published
    class function PolyTypeStr:String;override;
    function JsClassName:String;override;
  end;

  TGCopyright=class(TGOverlay,IJsClassWrapper,IHidable)
  private
    FminZoom: Integer;
    Fid: Integer;
    Fbounds: TGLatLngBounds;
    Ftext: String;
    procedure Setbounds(const Value: TGLatLngBounds);
    procedure Setid(const Value: Integer);
    procedure SetminZoom(const Value: Integer);
    procedure Settext(const Value: String);
  published
    property id 	    : Integer        read Fid write Setid; // A unique identifier for this copyright information.
    property minZoom 	: Integer        read FminZoom write SetminZoom; // The lowest zoom level at which this information applies.
    property bounds 	: TGLatLngBounds read Fbounds write Setbounds; // The region to which this information applies.
    property text 	  : String         read Ftext write Settext; // The text of the copyright message.
    constructor Create (aId : Integer; aBounds:TGLatLngBounds;aMinZoom:Integer;aText:String);
  end;

  TGoogleMaps=class(TPanel)
  private
    FWebBrowser:TWebBrowser;
    FhasEnd: Boolean;
    FhasStart: Boolean;
    FLogLines: TStrings;
    FOverlays: TOverlayList;
    FMapType: TGoogleMapType;
    FLatLngCenter: TGLatLng;
    FEnableDoubleClickZoom: Boolean;
    FEnableContinuousZoom: Boolean;
    FStatusPanel: TPanel;
    FJsVarName: String;
    procedure LoadHTML(URL:String);
    procedure SetLogLines(const Value: TStrings);
    procedure SetOverlays(const Value: TOverlayList);
    procedure Init;
    procedure SaveGoogleMapsHtml(const aFileName:String);
    procedure SetLatLngCenter(const Value: TGLatLng);
    procedure SetEnableContinuousZoom(const Value: Boolean);
    procedure SetEnableDoubleClickZoom(const Value: Boolean);
    function GetLatLngCenter: TGLatLng;
    property WebBrowser : TWebBrowser read FWebBrowser write FWebBrowser;
    procedure SetMapType(AMapType:TGoogleMapType);
    procedure SaveHTML(const FileName:String);
    function GetHTML: String;
    property hasStart     : Boolean read FhasStart write FhasStart;
    property hasEnd       : Boolean read FhasEnd write FhasEnd;
    procedure SetStatusPanel(const Value: TPanel);
    procedure SetJsVarName(const Value: String);
    property DragKind;
    property DragMode;
    property DockSite;
    property Ctl3D;
    property BiDiMode;
    property AutoSize;
    property HelpContext;
    property HelpKeyword;
    property HelpType;
    property Owner;
    property ParentBackground;
    property ParentBiDiMode;
    property ParentCtl3D;
    property Showing;
    property UseDockManager;
    property VerticalAlignment;
    property WheelAccumulator;



  public
    constructor Create(AOwner: TComponent);override;
    destructor Destroy;override;
    procedure SetCenter(Lat,Lng,Alt:Double;doPan:Boolean=false);overload;
    procedure SetCenter(Lat,Lng:Double;doPan:Boolean=false);overload;
    procedure SetCenter(LatLng:TGLatLng;doPan:Boolean=false);overload;
    procedure HandleOnResize(Sender:TObject);
    function GetJsValue(aJavaScript:String):OleVariant;
    property HTML: String read GetHTML;
    procedure CheckResize;
  published
    property StatusPanel  : TPanel read FStatusPanel write SetStatusPanel;
    property LogLines     : TStrings read FLogLines write SetLogLines;
    property Overlays     : TOverlayList read FOverlays write SetOverlays;
    property LatLngCenter : TGLatLng read GetLatLngCenter write SetLatLngCenter;
    property EnableContinuousZoom:Boolean read FEnableContinuousZoom write SetEnableContinuousZoom default true;
    property EnableDoubleClickZoom:Boolean read FEnableDoubleClickZoom write SetEnableDoubleClickZoom default true;
    property MapType:TGoogleMapType read FMapType write SetMapType;
    property JsVarName:String read FJsVarName write SetJsVarName;
    procedure AddControl(ControlType:TGoogleMapControl);
    procedure AddStartMarker;
    procedure AddEndMarker;
    procedure AddMarker(Lat,Lon:Double);
    procedure AddPolygon(GPolygon:TGPolygon);
    procedure AddOverlay(aOverlay:TGOverlay);
    procedure RemoveOverlay(aOverlay:TGOverlay);
    procedure RemoveOverlayByIndex(Index:Integer);
    procedure ClearOverlays;
    procedure SwapBeginEndMarkers;
    procedure GetDirections;
    procedure ShowAddress(const Street,City,State,Country:String);
    procedure openInfoWindow(aLatlng : TGLatLng; aHTML:String);
    procedure closeInfoWindow;
    procedure ExecJavaScript(const aScript:String);
    procedure WebBrowserDocumentComplete(ASender: TObject; const pDisp: IDispatch; var URL: OleVariant);
    procedure OnMouseOver;

    property Align;
    property OnClick;
    property OnCanResize;
    property OnResize;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnDblClick;
    property OnMouseWheel;
    property OnMouseWheelDown;
    property OnMouseWheelUp;
    property OnConstrainedResize;
    // property OnGetSiteInfo;
    property Anchors;
    property BorderWidth;
    property BoundsRect;
    property BorderStyle;
    property Brush;
    property ClientRect;
    property Canvas;
    property Color;
    property Enabled;
    property Action;
    property BevelEdges;
    property BevelKind;
    property BevelOuter;
    property BevelWidth;
    property ShowHint;
    property Visible;
    property Caption;
  end;

  TObjectProcedure = procedure of object;

  TEventObject = class(TInterfacedObject, IDispatch)
  private
    FOnEvent: TObjectProcedure;
  protected
    function GetTypeInfoCount(out Count: Integer): HResult; stdcall;
    function GetTypeInfo(Index, LocaleID: Integer; out TypeInfo): HResult; stdcall;
    function GetIDsOfNames(const IID: TGUID; Names: Pointer; NameCount, LocaleID: Integer; DispIDs: Pointer): HResult; stdcall;
    function Invoke(DispID: Integer; const IID: TGUID; LocaleID: Integer; Flags: Word; var Params; VarResult, ExcepInfo, ArgErr: Pointer): HResult; stdcall;
  public
    constructor Create(const OnEvent: TObjectProcedure) ;
    property OnEvent: TObjectProcedure read FOnEvent write FOnEvent;
  end;

  // todo: create a PlaceMark class to be able to natively handle Google Earth KML responses



procedure Register;

{$R uGoogleMaps.dcr}

implementation

uses
  Graphics,Windows;

procedure Register;
begin
  RegisterComponents('ARS', [TGoogleMaps]);
end;


function ColorToHtml(DColor:TColor):string;
var
  tmpRGB : TColor;
begin
  tmpRGB := ColorToRGB(DColor) ;
  Result:=Format('#%.2x%.2x%.2x',
                 [GetRValue(tmpRGB),
                  GetGValue(tmpRGB),
                  GetBValue(tmpRGB)]) ;
end;


{ TEventObject }

constructor TEventObject.Create(const OnEvent: TObjectProcedure) ;
begin
   inherited Create;
   FOnEvent := OnEvent;
end;

function TEventObject.GetIDsOfNames(const IID: TGUID; Names: Pointer; NameCount, LocaleID: Integer; DispIDs: Pointer): HResult;
begin
  Result := E_NOTIMPL;
end;

function TEventObject.GetTypeInfo(Index, LocaleID: Integer; out TypeInfo): HResult;
begin
  Result := E_NOTIMPL;
end;

function TEventObject.GetTypeInfoCount(out Count: Integer): HResult;
begin
  Result := E_NOTIMPL;
end;

function TEventObject.Invoke(DispID: Integer; const IID: TGUID; LocaleID: Integer; Flags: Word; var Params; VarResult, ExcepInfo, ArgErr: Pointer): HResult;
begin
   if (DispID = DISPID_VALUE) then
   begin
     if Assigned(FOnEvent) then FOnEvent;
     Result := S_OK;
   end
   else Result := E_NOTIMPL;
end;


{ TGoogleMaps }




procedure TGoogleMaps.Init;
var
  GoogleMapsFullFileName : String;
begin
  WebBrowser.OnDocumentComplete := WebBrowserDocumentComplete;
  GoogleMapsFullFileName := IncludeTrailingPathDelimiter(ExtractFilePath(Application.ExeName))+GoogleMapsFileName;
  SaveGoogleMapsHtml(GoogleMapsFullFileName);
  if FileExists(GoogleMapsFullFileName) then
    LoadHTML('file://' + GoogleMapsFullFileName);

  FOverlays := TOverlayList.Create;
end;

constructor TGoogleMaps.Create(AOwner: TComponent);
begin
  inherited;

  Color := 0;

  FStatusPanel           := TPanel.Create(self);
  FStatusPanel.Left      := 0;
  FStatusPanel.Top       := 0;
  FStatusPanel.Height    := 25;
  FStatusPanel.Parent    := Self;
  FStatusPanel.Align     := alBottom;
  FStatusPanel.Alignment := taLeftJustify;
  FStatusPanel.Show;

  FLatLngCenter := TGLatLng.Create(52,5);

  FLogLines := TStringList.Create;
  FEnableDoubleClickZoom := True;
  FEnableContinuousZoom  := True;
  FWEbBrowser := TWebBrowser.Create(self);
  TWinControl(FWEbBrowser).Parent := Self;
  FWebBrowser.Align := alClient;
  FWebBrowser.Show;

  JsVarName := 'map';

  Init;
end;


destructor TGoogleMaps.Destroy;
begin
  FreeAndNil(FOverlays);
  FreeAndNil(FLatLngCenter);
  FreeAndNil(FLogLines);
  inherited;
end;

procedure TGoogleMaps.ExecJavaScript(const aScript: String);
begin
  if (WebBrowser.ReadyState <> READYSTATE_COMPLETE) then
    exit;

  if Assigned(WebBrowser) then
    if Assigned(WebBrowser.Document) then
      (WebBrowser.Document as IHTMLDocument2).parentWindow.execScript(aScript, 'JavaScript');
end;

procedure TGoogleMaps.AddControl(ControlType: TGoogleMapControl);
begin
  ExecJavaScript(Format('addControl(%d);',[Integer(ControlType)]));
end;

procedure TGoogleMaps.AddStartMarker;
begin
  if not hasStart then
  begin
    ExecJavaScript('createStartMarker();');
    hasStart:=true;
  end
  else
    ExecJavaScript('centerMarker(start_marker);');
end;

procedure TGoogleMaps.AddEndMarker;
begin
  if not hasEnd then
  begin
    ExecJavaScript('createEndMarker();');
    hasEnd:=true;
  end
  else
    ExecJavaScript('centerMarker(end_marker);');
end;

procedure TGoogleMaps.addMarker(Lat, Lon: Double);
var
  Marker:TGMarker;
begin
//  JavaScript(format('createMarker(%g,%g)',[Lat,Lon]));
  AddOverlay(TGMarker.Create(TGLatLng.Create(Lat,Lon)));
end;

procedure TGoogleMaps.AddOverlay(aOverlay:TGOverlay);
begin
  FOverlays.Add(aOverlay);
  ExecJavaScript('var '+aOverlay.JsVarName + '=' + aOverlay.ToJavaScript+ ';');
  ExecJavaScript(JsVarName+'.addOverlay('+aOverlay.JsVarName+')');
end;

procedure TGoogleMaps.AddPolygon(GPolygon: TGPolygon);
begin
  AddOverlay(GPolygon);
end;

procedure TGoogleMaps.GetDirections;
begin
  ExecJavaScript('directions();');
end;


procedure TGoogleMaps.LoadHTML(URL: String);
begin
  WebBrowser.Navigate(URL);
end;

procedure TGoogleMaps.SetCenter(Lat, Lng, Alt: Double;doPan:Boolean=False);
var
  Operation:String;
begin
  if DoPan then
    Operation := 'panTo'
  else
    Operation := 'setCenter';

  FLatLngCenter.Lat := Lat;
  FLatLngCenter.Lng := Lng;
  ExecJavaScript(Format(JsVarName+'.'+Operation+'(new GLatLng(%g,%g), %g);',[Lat,Lng,Alt]));
  FStatusPanel.Caption := FLatLngCenter.ToString;

end;

procedure TGoogleMaps.SetCenter(Lat, Lng: Double;doPan:Boolean=False);
var
  Operation:String;
begin
  if DoPan then
    Operation := 'panTo'
  else
    Operation := 'setCenter';

  FLatLngCenter.Lat := Lat;
  FLatLngCenter.Lng := Lng;
  ExecJavaScript(Format(JsVarName+'.'+Operation+'(new GLatLng(%g,%g));',[Lat,Lng]));
  FStatusPanel.Caption := FLatLngCenter.ToString;
end;

procedure TGoogleMaps.SetEnableContinuousZoom(const Value: Boolean);
begin
//  if Value<>FEnableContinuousZoom then
  begin
    FEnableContinuousZoom := Value;
    if FEnableContinuousZoom then
      ExecJavaScript(JsVarName+'.enableContinuousZoom();')
    else
      ExecJavaScript(JsVarName+'.disableContinuousZoom();');
  end;
end;

procedure TGoogleMaps.SetEnableDoubleClickZoom(const Value: Boolean);
begin
//  if Value<>FEnableDoubleClickZoom then
  begin
    FEnableDoubleClickZoom := Value;
    if FEnableContinuousZoom then
      ExecJavaScript(JsVarName+'.enableDoubleClickZoom();')
    else
      ExecJavaScript(JsVarName+'.disableDoubleClickZoom();');
  end;
end;

procedure TGoogleMaps.SetLatLngCenter(const Value: TGLatLng);
begin
  FLatLngCenter := Value;
  SetCenter(FLatLngCenter.Lat,FLatLngCenter.Lng);
end;

procedure TGoogleMaps.SetLogLines(const Value: TStrings);
begin
  FLogLines := Value;
end;

procedure TGoogleMaps.SetMapType(AMapType: TGoogleMapType);
begin
  FMapType := AMapType;
  ExecJavaScript(Format('setMapType(%d);',[Integer(FMapType)]));
end;

procedure TGoogleMaps.SetOverlays(const Value: TOverlayList);
begin
  FOverlays := Value;
end;

procedure TGoogleMaps.ShowAddress(const Street, City, State, Country: String);
begin
  ExecJavaScript(Format('showAddress("%s, %s, %s, %s");',[Street,City,State,Country]));
end;

procedure TGoogleMaps.SwapBeginEndMarkers;
begin
  ExecJavaScript('swap();');
end;

procedure TGoogleMaps.WebBrowserDocumentComplete(ASender: TObject;
  const pDisp: IDispatch; var URL: OleVariant);
begin
  if Assigned(FWebBrowser.Document) then
  begin
    (FWebBrowser.Document as IHTMLDocument2).onmouseover := (TEventObject.Create(OnMouseOver) as IDispatch) ;
//    (FWebBrowser.Document as IHTMLDocument2).parentWindow.alert((FWebBrowser.Document as IHTMLDocument2).parentWindow.toString);
  end;
end;

procedure TGoogleMaps.OnMouseOver;
var
  element : IHTMLElement;
begin
  element := (FWebBrowser.Document as IHTMLDocument2).parentWindow.event.srcElement;

  if not Assigned(FLogLines) then
    Exit;

  FLogLines.Clear;
  if LowerCase(element.tagName) = 'a' then
  begin
    FLogLines.Add('LINK info...');
    FLogLines.Add(Format('HREF : %s',[element.getAttribute('href',0)]));
  end
  else if LowerCase(element.tagName) = 'img' then
  begin
    FLogLines.Add('IMAGE info...');
    FLogLines.Add(Format('SRC : %s',[element.getAttribute('src',0)]));
  end
  else
  begin
    FLogLines.Add(Format('TAG : %s',[element.tagName]));
  end;
end;

procedure TGoogleMaps.openInfoWindow(aLatlng: TGLatLng; aHTML: String);
begin
  ExecJavaScript(JsVarName + '.openInfoWindow('+aLatlng.ToJavaScript+', "'+aHTML+'")');
end;

procedure TGoogleMaps.RemoveOverlay(aOverlay: TGOverlay);
begin
  FOverlays.Remove(aOverlay);
  ExecJavaScript(JsVarName+'.removeOverlay('+aOverlay.JsVarName+');');
  ExecJavaScript('delete '+aOverlay.JsVarName+';');
end;

procedure TGoogleMaps.RemoveOverlayByIndex(Index: Integer);
begin
  RemoveOverlay(Overlays.Items[index]);
end;

function TGoogleMaps.GetHTML: String;
begin
  Result := '';
  {$include uGooglemaps.Inc}
end;

procedure TGoogleMaps.SaveGoogleMapsHtml(const aFileName:String);
var
 ResStream: TResourceStream;
 FileStream: TFileStream;
begin
  ResStream := TResourceStream.Create(hInstance, 'GOOGLE_MAPS_HTML', RT_RCDATA) ;
  try
    FileStream := TFileStream.Create(aFileName, fmCreate) ;
    try
      FileStream.CopyFrom(ResStream, 0) ;
    finally
      FileStream.Free;
    end;
  finally
    ResStream.Free;
  end;
end;

procedure TGoogleMaps.SaveHTML(const FileName:String);
var Strings:TStringList;
begin
  Strings := TStringList.Create;
  try
    Strings.Text := HTML;
    Strings.SaveToFile(FileName);
  finally
    Strings.Free;
  end;
end;

function TGoogleMaps.GetJsValue(aJavaScript: String): OleVariant;
  function GetFormByNumber(document: IHTMLDocument2;
      formNumber: integer): IHTMLFormElement;
  var
    forms: IHTMLElementCollection;
  begin
    forms := document.Forms as IHTMLElementCollection;
    if formNumber < forms.Length then
      result := forms.Item(formNumber,'') as IHTMLFormElement
    else
      result := nil;
  end;

  function GetFieldValue(fromForm: IHTMLFormElement; const fieldName: string): string;
  var
    field: IHTMLElement;
    inputField: IHTMLInputElement;
    selectField: IHTMLSelectElement;
    textField: IHTMLTextAreaElement;
  begin
    field := fromForm.Item(fieldName,'') as IHTMLElement;
    if not Assigned(field) then
      result := ''
    else
    begin
      if field.tagName = 'INPUT' then
      begin
        inputField := field as IHTMLInputElement;
        result := inputField.value
      end
      else if field.tagName = 'SELECT' then
      begin
        selectField := field as IHTMLSelectElement;
        result := selectField.value
      end
      else if field.tagName = 'TEXTAREA' then
      begin
        textField := field as IHTMLTextAreaElement;
        result := textField.value;
      end;
    end
  end;


var
  document: IHTMLDocument2;
  theForm: IHTMLFormElement;

begin
  ExecJavaScript('ReturnValue('+aJavaScript+')');

  document := WebBrowser.Document as IHTMLDocument2;
  theForm := GetFormByNumber(WebBrowser.Document as IHTMLDocument2,0);
  Result := GetFieldValue(theForm,'retVal');
end;

function TGoogleMaps.GetLatLngCenter: TGLatLng;
begin
//  FLatLngCenter.Lat := GetJsValue(JsVarName+'.getCenter().lat()');
//  FLatLngCenter.Lng := GetJsValue(JsVarName+'.getCenter().lng()');
  Result            := FLatLngCenter;
end;

procedure TGoogleMaps.SetCenter(LatLng: TGLatLng; doPan: Boolean);
begin
  SetCenter(LatLng.Lat,LatLng.Lng,DoPan);
end;

procedure TGoogleMaps.CheckResize;
begin
  ExecJavaScript(JsVarName+'.checkResize();');
end;

procedure TGoogleMaps.HandleOnResize(Sender: TObject);
begin
  CheckResize;
end;

procedure TGoogleMaps.ClearOverlays;
begin
  FOverlays.Clear;
  ExecJavaScript(JsVarName+'.clearOverlays();');
end;

procedure TGoogleMaps.closeInfoWindow;
begin
  ExecJavaScript(JsVarName+'.closeInfoWindow();');
end;

procedure TGoogleMaps.SetStatusPanel(const Value: TPanel);
begin
  FStatusPanel := Value;
end;

procedure TGoogleMaps.SetJsVarName(const Value: String);
begin
  FJsVarName := Value;
end;

{ TGPolygon }

function TGPolygon.AddPoint(const aGLatLng: TGLatLng): integer;
begin
  Result := -1;
  if (Count>0) and (aGLatLng.Equals(FCoordinates[High(FCoordinates)])) then
    Exit;

  SetLength(FCoordinates,Length(FCoordinates)+1);
  FCoordinates[High(FCoordinates)] := aGLatLng;
end;

function TGPolygon.JsClassName: String;
begin
  Result := 'GPolygon';
end;

procedure TGPolygon.Clear;
begin
  FreeAndNil(FSimplified);
  SetLength(FCoordinates,0);
end;

destructor TGPolygon.Destroy;
var
  I : integer;
begin
  FreeAndNil(FSimplified);
  for I := 0 to High(FCoordinates) do
    FreeAndNil(FCoordinates[I]);
  inherited;
end;

constructor TGPolygon.Create(const ACoordinates: array of TGLatLng);
var
  I :integer;
begin
  Color := clYellow;
  SetLength(FCoordinates,Length(ACoordinates));

  for I := 0 to High(ACoordinates) do
    FCoordinates[I] := ACoordinates[I];
end;

constructor TGPolygon.Create(const aPoints: array of TPointFloat2D);
var
  I :integer;
begin
  Color := clYellow;
  SetLength(FCoordinates,Length(APoints));
  for I := 0 to High(APoints) do
    FCoordinates[I] := TGLatLng.Create(APoints[I].Y,APoints[I].X);
end;


function TGPolygon.GetCount: Integer;
begin
  Result := Length(FCoordinates);
end;

function TGPolygon.GetSimplified: TGPolygon;
begin
  if (not Assigned(FSimplified)) or  FSimplified.IsDirty then
    FSimplified := getSimplifiedVersion;

  Result := FSimplified;
end;

function TGPolygon.getSimplifiedVersion(Tolerance:Double): TGPolygon;
var
  OrigAr,
  SimplifiedAr:TFloat2DPointAr;
  I: Integer;
begin

  SetLength( OrigAr, Count );

  for I := 0 to Count - 1 do
  begin
    OrigAr[I].X := FCoordinates[I].Lng;
    OrigAr[I].Y := FCoordinates[I].Lat;
  end;

  PolySimplifyFloat2D( Tolerance, OrigAr, SimplifiedAr );

  for I := 0 to High(SimplifiedAr) do
  begin
    OrigAr[I].X := FCoordinates[I].Lng;
    OrigAr[I].Y := FCoordinates[I].Lat;
  end;

  Result := TGPolygon.Create(SimplifiedAr);
end;

class function TGPolygon.PolyTypeStr: String;
begin
  { TODO 3 -oWouter -cfix : wrong type used here for now}
  Result := 'GPolyline';
end;

procedure TGPolygon.SetColor(const Value: TColor);
begin
  FColor := Value;
end;

procedure TGPolygon.SetOpacity(const Value: double);
begin
  FOpacity := Value;
end;

procedure TGPolygon.SetSimplified(const Value: TGPolygon);
begin
  FSimplified := Value;
end;

procedure TGPolygon.SetWeightPx(const Value: integer);
begin
  FWeightPx := Value;
end;

function TGPolygon.supportsHide: Boolean;
begin
  Result := True;
end;

function TGPolygon.ToJavaScript: String;
var
  I : Integer;
begin
  Result := ' new ' + PolyTypeStr + '(['#13#10;
  for I := 0 to High(FCoordinates) do
  begin
    Result := Result +  FCoordinates[I].ToJavaScript;
    if I<High(FCoordinates) then
      Result := Result + ','#13#10;
  end;
	Result := Result + '], "'+ColorToHtml(Color)+'", 5, 1, "#ff0000", 0.2)'#13#10;
end;

{ TGLatLng }

constructor TGLatLng.Create(aLat, aLng: Double);
begin
  FLat := aLat;
  Flng := aLng;
end;

function TGLatLng.Equals(const AGLatLng: TGLatLng): Boolean;
begin
  Result := (AGLatLng.Lat=Lat) and (AGLatLng.Lng=Lng);
end;

function TGLatLng.GetJsVarName: String;
begin
  Result := FJsVarName;
end;

function TGLatLng.JsClassName: String;
begin
  Result := 'GLatLng';
end;

procedure TGLatLng.SetJsVarName(const Value: String);
begin
  FJsVarName := Value;
end;

function TGLatLng.ToJavaScript: String;
begin
  DecimalSeparator := '.';
  Result := Format(' new GLatLng(%g,%g)',[ Lat, Lng ]);
end;

function TGLatLng.ToString: String;
begin
  Result := Format('(Lat:%g,Lng:%g)',[Lat,Lng])
end;

{ TGPolyLine }

function TGPolyLine.JsClassName: String;
begin
  Result := 'GPolyline';
end;

class function TGPolyLine.PolyTypeStr: String;
begin
  Result := 'GPolyline';
end;

{ TMarker }

destructor TGMarker.Destroy;
begin
  FreeAndNil(FCenter);
  inherited;
end;

function TGMarker.JsClassName: String;
begin
  Result := 'GMarker';
end;

procedure TGMarker.openInfoWindow(aContent:String);
begin
  GoogleMaps.ExecJavaScript(JsVarName+'.openInfoWindow("'+aContent+'");');
end;

procedure TGMarker.openInfoWindowHtml(aContent:String);
begin
  GoogleMaps.ExecJavaScript(JsVarName+'.openInfoWindowHtml("'+aContent+'");');
end;

procedure TGMarker.SetDraggingEnabled(const Value: Boolean);
begin
  FDraggingEnabled := Value;
end;

constructor TGMarker.Create(const aCenter: TGLatLng);
begin
  Center := aCenter;
end;


procedure TGMarker.setLatLng(const Value: TGLatLng);
begin
  FCenter := Value;
end;

function TGMarker.SupportsHide: Boolean;
begin
  Result := True;
end;

function TGMarker.ToJavaScript: String;
begin
  Result := 'new GMarker(' + Center.ToJavaScript + ', {icon: G_START_ICON, draggable: false})';
end;

{ TOverlayList }

function TOverlayList.Add(aGOverlay: TGOverlay): Integer;
begin
  aGOverlay.ID := AutoIncrementID;
  inc(AutoIncrementID);
  result := inherited Add(aGOverLay);
end;

procedure TOverlayList.Clear;
var
  i : integer;
begin
  for I := 0 to Count - 1 do
    if Assigned(GetItem(I)) then
      GetItem(I).Free;

  inherited;
end;

constructor TOverlayList.Create;
begin
end;

destructor TOverlayList.Destroy;
var
  i : integer;
begin
  for i:=0 to Count-1 do
    Items[I].Free;
end;

function TOverlayList.GetItems(Index: Integer): TGOverlay;
begin
  result := TGOverlay(inherited Items[Index]);
end;

procedure TOverlayList.SetItems(Index: Integer; const Value: TGOverlay);
begin
  inherited Items[Index] := Value;
end;

function TOverlayList.ToString: String;
var
  I: Integer;
begin
  for I := 0 to Count - 1 do
    Result := Result + Format('%d : %s (%s)',[I,Items[I].JsVarName,Items[I].JsClassName])+ #13#10;

  Result := Trim(Result);
end;

{ TGOverlay }

function TGOverlay.GetJsVarName: String;
begin
  FJsVarName := 'myOverlay_'+IntToStr(ID);
  Result := FJsVarName;
end;


procedure TGOverlay.hide;
begin
  inherited;
  if supportsHide then
    FGoogleMaps.ExecJavaScript(JsVarName + '.hide();');
end;

function TGOverlay.isHidden: Boolean;
begin
  Result := FGoogleMaps.GetJsValue(JsVarName + '.isHidden()');
end;

function TGOverlay.JsClassName: string;
begin
  Result := 'GOverlay';
end;

procedure TGOverlay.SetGoogleMaps(const Value: TGoogleMaps);
begin
  FGoogleMaps := Value;
end;

procedure TGOverlay.SetID(const Value: Integer);
begin
  FID := Value;
end;

procedure TGOverlay.SetJsVarName(const Value: String);
begin

end;

procedure TGOverlay.SetName(const Value: String);
begin
  FName := Value;
end;

procedure TGOverlay.show;
begin
  FGoogleMaps.ExecJavaScript(JsVarName + '.show();');
end;

function TGOverlay.supportsHide: Boolean;
begin
  Result := false;
end;

{ TGBounds }

destructor TGBounds.Destroy;
begin
  FreeAndNil(FMin);
  FreeAndNil(FMid);
  FreeAndNil(FMax);
  inherited;
end;

function TGBounds.Equals(aGBounds: TGBounds): Boolean;
begin
  Result :=
  (
    (minX = aGBounds.minX) and
    (minY = aGBounds.minY) and
    (maxX = aGBounds.maxX) and
    (maxY = aGBounds.maxY)
  );
end;

function TGBounds.GetJsVarName: String;
begin
  Result := FJsVarName;
end;

function TGBounds.GetMax: TGLatLng;
begin
  Result := TGLatLng.Create(maxY,MaxX);
end;

function TGBounds.GetMid: TGLatLng;
begin
  if not Assigned(FMin) then
    FMid := TGLatLng.Create(minX+((maxX-minX)/2),minY+((maxY-minY)/2));

  Result := FMid;
end;

function TGBounds.GetMin: TGLatLng;
begin
  Result := TGLatLng.Create(minY,MinX);
end;

function TGBounds.JsClassName: String;
begin
  Result := 'GBounds';
end;

procedure TGBounds.SetJsVarName(const Value: String);
begin
  FJsVarName := Value;
end;

function TGBounds.ToJavaScript: String;
begin
  // @@@ not implemented
end;

function TGBounds.ToString: String;
begin
  // @@@ not yet implemented
end;

{ TGeoXML }

constructor TGGeoXml.Create(const aUrlOfXml: String);
begin
  FUrlOfXml:= aUrlOfXml;
end;

destructor TGGeoXml.Destroy;
begin
  // @@@ nothing to clean up
  inherited;
end;

procedure TGGeoXml.gotoDefaultViewport(Map: TGoogleMaps);
begin
  FGoogleMaps.ExecJavaScript(JsVarName + '.gotoDefaultViewport('+ FGoogleMaps.JsVarName +');');
end;

function TGGeoXml.JsClassName: String;
begin
  Result := 'GGeoXml';
end;

procedure TGGeoXml.SetUrlOfXml(const Value: String);
begin
  FUrlOfXml := Value;
end;

function TGGeoXml.supportsHide: Boolean;
begin
  Result := True;
end;

function TGGeoXml.ToJavaScript: String;
begin
  FGoogleMaps.ExecJavaScript('var '+JsVarName+' = new '+JsClassName+'("'+FUrlOfXml+'", function() { if ('+JsVarName+'.loadedCorrectly()) {'+JsVarName+'.gotoDefaultViewport('+GoogleMaps.JsVarName+');}})');
  Result := JsVarName;
end;

{ TGCopyright }

constructor TGCopyright.Create(aId: Integer; aBounds: TGLatLngBounds;
  aMinZoom: Integer; aText: String);
begin
  Fid      := aID;
  FBounds  := aBounds;
  FMinZoom := aMinZoom;
  FText    := aText;
end;

procedure TGCopyright.Setbounds(const Value: TGLatLngBounds);
begin
  Fbounds := Value;
end;

procedure TGCopyright.Setid(const Value: Integer);
begin
  Fid := Value;
end;

procedure TGCopyright.SetminZoom(const Value: Integer);
begin
  FminZoom := Value;
end;

procedure TGCopyright.Settext(const Value: String);
begin
  Ftext := Value;
end;

{ TGLatLngBounds }


function TGLatLngBounds.contains(aLatLng: TGLatLng): Boolean;
begin

  // function(a){return this.va.contains(a.pd())&&this.fa.contains(a.qd())}

  // TGLatLngBounds.contains(aLatLng: TGLatLng): Boolean;
end;

function TGLatLngBounds.containsBounds(aGLatLngBounds: TGLatLngBounds): Boolean;
begin
  //TGLatLngBounds.containsBounds=function(a){return this.va.jh(a.va)&&this.fa.jh(a.fa)}

end;

function TGLatLngBounds.containsLatLng(aLatLng: TGLatLng): Boolean;
begin
//TGLatLngBounds.containsLatLng=function(a){return this.va.contains(a.pd())&&this.fa.contains(a.qd())}

end;

constructor TGLatLngBounds.Create(sw, ne: TGLatLng);
begin

end;

destructor TGLatLngBounds.Destroy;
begin

  inherited;
end;

function TGLatLngBounds.Equals(aGLatLngBounds: TGLatLngBounds): Boolean;
begin
  Result :=
    NorthEast.Equals(aGLatLngBounds.NorthEast) and
    SouthWest.Equals(aGLatLngBounds.SouthWest);
end;

procedure TGLatLngBounds.extend(aLatLng: TGLatLng);
begin
{
  NorthEast. va.extend(a.pd());
  SouthWest.Extend(
  this.fa.extend(aLatLng.qd())
  }
end;

function TGLatLngBounds.getCenter: TGLatLng;
begin
  //TGLatLngBounds.getCenter=function(){return K.fromRadians(this.va.center(),this.fa.center())}
end;

function TGLatLngBounds.getNorthEast: TGLatLng;
begin
  //TGLatLngBounds.getNorthEast=function(){return K.fromRadians(this.va.hi,this.fa.hi)}
end;

function TGLatLngBounds.getSouthWest: TGLatLng;
begin
  //TGLatLngBounds.getSouthWest=function(){return K.fromRadians(this.va.lo,this.fa.lo)}
end;

function TGLatLngBounds.intersects(aGLatLngBounds: TGLatLngBounds): Boolean;
begin
  //TGLatLngBounds.intersects=function(a){return this.va.intersects(a.va)&&this.fa.intersects(a.fa)}
end;

function TGLatLngBounds.isEmpty: Boolean;
begin
  //TGLatLngBounds.isEmpty=function(){return this.va.$()||this.fa.$()}
end;

function TGLatLngBounds.isFullLat: Boolean;
begin
  //TGLatLngBounds.isFullLat=function(){return this.va.hi>=Bc/2&&this.va.lo<=-Bc/2}
end;

function TGLatLngBounds.isFullLng: Boolean;
begin
  //TGLatLngBounds.isFullLng=function(){return this.fa.Yh()}
end;

procedure TGLatLngBounds.setNorthEast(const Value: TGLatLng);
begin

end;

procedure TGLatLngBounds.setSouthWest(const Value: TGLatLng);
begin

end;

function TGLatLngBounds.toSpan: TGLatLng;
begin
  //TGLatLngBounds.toSpan=function(){return K.fromRadians(this.va.span(),this.fa.span(),true)}
end;

function TGLatLngBounds.ToString: String;
begin

end;

{ TGInfoWindow }

constructor TGInfoWindow.Create(const aCenter: TGLatLng);
begin

end;

destructor TGInfoWindow.Destroy;
begin

  inherited;
end;


function TGInfoWindow.JsClassName: String;
begin
  Result := 'GInfoWindow';
end;

procedure TGInfoWindow.Maximize;
begin
  //
end;

procedure TGInfoWindow.Restore;
begin
  //
end;


procedure TGInfoWindow.SetHTML(const Value: String);
begin
  FHTML := Value;
end;

function TGInfoWindow.supportsHide: Boolean;
begin
  Result := True;
end;

function TGInfoWindow.ToJavaScript: String;
//var
//  KmlObject:IKmlObject;
begin
 // map.openInfoWindow(map.getCenter(),                   document.createTextNode("Hello, world"));
  Result := GoogleMaps.JsVarName + '.getInfoWindowHtml();';

end;

end.
